/*
 * Copyright (c) 2001 Matthew Feldt. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided the copyright notice above is
 * retained.
 *
 * THIS SOFTWARE IS PROVIDED ''AS IS'' AND WITHOUT ANY EXPRESSED OR
 * IMPLIED WARRANTIES.
 */

/**
 * MetarRun.java
 *
 * Class to read metar data from ftp or file and present it back to the user.
 *
 * @author Matthew Feldt <developer@feldt.com>
 * @version 0.1, 01/25/2005
 */
package com.feldt.metar;

import jargs.gnu.CmdLineParser;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;

public class MetarRun {

	public static void main(String[] args) {
		final String usage = "Usage: java -jar metar-app-1.0.jar "
			+ "[{-p,--print-details}] [{-f,--ftp}] " + System.getProperty("line.separator")
			+ "       [{-i,--input-file} filename] [--help] station";
		Metar metar = null;
		String station;

		CmdLineParser parser = new CmdLineParser();
		CmdLineParser.Option detailOption = parser.addBooleanOption('p', "print-details");
		CmdLineParser.Option ftpOption = parser.addBooleanOption('f', "ftp");
		CmdLineParser.Option fileOption = parser.addStringOption('i', "input-file");
		CmdLineParser.Option helpOption = parser.addBooleanOption("help");
		
		try {
			parser.parse(args);
		} catch (CmdLineParser.OptionException e) {
			System.err.println(e.getMessage());
			System.err.println(usage);
			System.exit(-1);
		}
		
		Boolean helpValue = (Boolean)parser.getOptionValue(helpOption);
		if (helpValue != null && helpValue.booleanValue()) {
			System.out.println();
			System.out.println(usage);
			System.out.println();
			System.out.println("  Station names can be found in metar.properties or from the"); 
			System.out.println("  offical source at: http://weather.noaa.gov/data/nsd_bbsss.txt");
			System.out.println();
			System.out.println("  -f,--ftp             fetch metar data via ftp");
			System.out.println("  -i,--input-file      read metar data from the specified file");
			System.out.println("  -p,--print-details   print parsed and converted metar data");
			System.out.println("  --help               this handy and helpful message");
			System.out.println();
			System.exit(0);
		}

		String[] remainingArgs = parser.getRemainingArgs();
		if (remainingArgs.length != 1) {
			System.err.println(usage);
			System.exit(-1);
		}
		station = remainingArgs[0].toUpperCase();
		
		try {
			Boolean detailValue = (Boolean) parser.getOptionValue(detailOption);
			if (detailValue == null) {
				detailValue = Boolean.FALSE;
			}
			
			Boolean ftpValue = (Boolean)parser.getOptionValue(ftpOption);
			if (ftpValue == null) {
				ftpValue = Boolean.FALSE;
			}
			
			String inputFile = (String)parser.getOptionValue(fileOption);
			if (inputFile != null) {	// read a file looking for a line that begins with 
									// the station and parse that line as metar data 
									// then display the result
				boolean foundMetarData = false;
				String data, metarDate = null;
				String matcher = "\\d{4}/\\d{2}/\\d{2} \\d{2}:\\d{2}";
				
				File f = new File(inputFile);
				if (!f.isFile() || !f.canRead()) {
					throw new Exception(f.getPath()
						+ ": does not exist or can not be read.");
				}
				
				// read file searching for METAR observation data
				BufferedReader in = new BufferedReader(new FileReader(f));

				while ((data = in.readLine()) != null) {
					if (data.matches(matcher)) {
						metarDate = data;
					}
					
					if (data.startsWith(station) && metarDate != null) {
						foundMetarData = true;

						// remove "=<BR>" from the end of the line
						if (data.substring(data.length() - 5).equals("=<BR>")) {
							data = data.substring(0, data.length() - 5);
						}

						// create a Metar object from the discovered metar string
						metar = Metar.parse(station, data, metarDate);
						
						// there can be more than one METAR data line in a file
						// but for this example, i'm only interested in the first
						break;
					}
				}
				
				if (foundMetarData && metar != null) {
					if (detailValue.booleanValue()) {
						System.out.println(metar.getFormattedMetar());
					} else {
						System.out.println(metar.toString());
					}
				} else {
					System.out.println("No metar data file in file.");
				}
						
			} else if (ftpValue.booleanValue()) { // fetch metar data via ftp and display result	
				MetarFtp metarFtp = MetarFtp.parse(station);
				metar = Metar.parse(station, metarFtp.getObservation(), metarFtp.getObservationDateAsString());
				if (metar != null) {
					if (detailValue.booleanValue()) {
						System.out.println(metar.getFormattedMetar());
					} else {
						System.out.println(metar.toString());
					}
				}
			}
		} catch (Exception e) {
			System.err.println(e.getMessage());
		}
	}
}