// Copyright 2022, 2023, 2024 Global Virtual Airlines Group. All Rights Reserved.
package org.deltava.beans.simbrief;

import java.util.*;
import java.time.Instant;

import org.jdom2.Document;

import org.deltava.beans.*;
import org.deltava.beans.flight.ETOPS;
import org.deltava.beans.schedule.Airport;

import org.deltava.util.XMLUtils;

/**
 * A bean to store a SimBrief briefing package.
 * @author Luke
 * @version 11.2
 * @since 10.3
 */

public class BriefingPackage extends DatabaseBean {

	private final PackageFormat _fmt;
	private String _sbID;
	private String _sbUserID;
	private String _sbReqID;
	private int _releaseVersion;
	private Instant _createdOn;
	private String _tailCode;
	private String _airframeID;
	
	private ETOPS _etops = ETOPS.ETOPS60;
	private GeoLocation _etopsMidpoint;
	private final List<Airport> _alternates = new ArrayList<Airport>();
	private final List<Airport> _etopsAlts = new ArrayList<Airport>();
	
	private int _airac;
	private String _url;
	private String _cruiseAlt;
	private String _route;
	
	private String _rwyD;
	private String _rwyA;
	
	private int _baseFuel;
	private int _taxiFuel;
	private int _enrouteFuel;
	private int _altFuel;
	
	private int _pax;
	private int _baggageWeight;
	private int _bagWeight;
	private int _cargoWeight;
	
	private String _baseFPURL;
	private final Collection<FlightPlan> _plans = new TreeSet<FlightPlan>();
	
	private Document _xml;
	
	/**
	 * Creates the bean.
	 * @param id the Flight Report database ID
	 * @param fmt the PackageFormat
	 */
	BriefingPackage(int id, PackageFormat fmt) {
		super();
		setID(id);
		_fmt = fmt;
	}

	/**
	 * Returns the SimBrief ID for this package.
	 * @return the static ID
	 */
	public String getSimBriefID() {
		return _sbID;
	}
	
	/**
	 * Returns the SimBrief User ID for this package.
	 * @return the SimBrief user ID
	 */
	public String getSimBriefUserID() {
		return _sbUserID;
	}
	
	/**
	 * Returns the SimBrief request ID.
	 * @return the SimBrief request ID
	 */
	public String getRequestID() {
		return _sbReqID;
	}
	
	/**
	 * Returns the briefing package format.
	 * @return the format
	 */
	public PackageFormat getFormat() {
		return _fmt;
	}
	
	/**
	 * Returns the briefing release version.
	 * @return the version
	 */
	public int getReleaseVersion() {
		return _releaseVersion;
	}

	/**
	 * Returns the generation date of this package.
	 * @return the date/time the package was generated by SimBrief
	 */
	public Instant getCreatedOn() {
		return _createdOn;
	}
	
	/**
	 * Returns the AIRAC version used for this route.
	 * @return the AIRAC version
	 */
	public int getAIRAC() {
		return _airac;
	}
	
	/**
	 * Returns the arrival alternate Airports.
	 * @return a List of Airports
	 */
	public List<Airport> getAlternates() {
		return _alternates;
	}
	
	/**
	 * Returns the cruise altitude.
	 * @return the altitude
	 */
	public String getCruiseAltitude() {
		return _cruiseAlt;
	}

	/**
	 * Returns the flight route.
	 * @return the flight route
	 */
	public String getRoute() {
		return _route;
	}
	
	/**
	 * Returns the aircraft tail code.
	 * @return the tail code
	 */
	public String getTailCode() {
		return _tailCode;
	}
	
	/**
	 * Returns the SimBrief custom airframe ID.
	 * @return the aiframe ID
	 */
	public String getAirframeID() {
		return _airframeID;
	}
	
	/**
	 * Returns the route ETOPS rating.
	 * @return the ETOPS rating
	 */
	public ETOPS getETOPS() {
		return _etops;
	}
	
	/**
	 * Returns the ETOPS diversion Airports.
	 * @return a List of Airports
	 */
	public List<Airport> getETOPSAlternates() {
		return _etopsAlts;
	}
	
	/**
	 * Returns the ETOPS equal time point location.
	 * @return a GeoLocation, or null
	 */
	public GeoLocation getETOPSMidpoint() {
		return _etopsMidpoint;
	}
	
	/**
	 * Returns the departure Runway.
	 * @return the departure runway name
	 */
	public String getRunwayD() {
		return _rwyD;
	}
	
	/**
	 * Returns the arrival Runway.
	 * @return the arrival runway name
	 */
	public String getRunwayA() {
		return _rwyA;
	}
	
	/**
	 * Returns the URL used to download the SimBrief package.
	 * @return the SimBrief URL
	 */
	public String getURL() {
		return _url;
	}
	
	/**
	 * Returns the base URL for downloadable Flight Plans.
	 * @return the URL
	 */
	public String getBasePlanURL() {
		return _baseFPURL;
	}
	
	/**
	 * Returns the Flight Plans in this briefing package.
	 * @return a Collection of FlightPlans
	 */
	public Collection<FlightPlan> getFlightPlans() {
		return _plans;
	}
	
	/**
	 * Returns the raw XML generated by SimBrief.
	 * @return the XML Document
	 */
	public Document getXML() {
		return _xml;
	}
	
	/**
	 * Returns the base fuel load.
	 * @return fuel the fuel in pounds
	 */
	public int getBaseFuel() {
		return _baseFuel;
	}

	/**
	 * Returns the taxi fuel load.
	 * @return fuel the fuel in pounds
	 */
	public int getTaxiFuel() {
		return _taxiFuel;
	}
	
	/**
	 * Returns the enroute fuel load.
	 * @return fuel the fuel in pounds
	 */
	public int getEnrouteFuel() {
		return _enrouteFuel;
	}
	
	/**
	 * Returns the alternate fuel load.
	 * @return fuel the fuel in pounds
	 */
	public int getAlternateFuel() {
		return _altFuel;
	}
	
	/**
	 * Returns the total fuel load.
	 * @return the fuel in pounds
	 */
	public int getTotalFuel() {
		return _baseFuel + _taxiFuel + _enrouteFuel + _altFuel;
	}
	
	/**
	 * Returns the HTML pilot briefing text.
	 * @return the HTML text
	 */
	public String getBriefingText() {
		return XMLUtils.getChildText(_xml.getRootElement(), "text", "plan_html");
	}
	
	/**
	 * Returns the number of passengers booked for this flight.
	 * @return the number of passengers
	 */
	public int getPax() {
		return _pax;
	}
	
	/**
	 * Returns the total baggage weight for this flight.
	 * @return the weight in pounds
	 */
	public int getBaggageWeight() {
		return _baggageWeight;
	}
	
	/**
	 * Returns the per-passenger baggage weight for this flight.
	 * @return the weight in pounds
	 */
	public int getBagWeight() {
		return _bagWeight;
	}
	
	/**
	 * Returns the additional cargo weight for this flight.
	 * @return the weight in pounds
	 */
	public int getCargoWeight() {
		return _cargoWeight;
	}
	
	/**
	 * Adds a flight plan to the briefing package.
	 * @param fp a FlightPlan
	 */
	public void addPlan(FlightPlan fp) {
		_plans.add(fp);
	}
	
	/**
	 * Updates the generation date of this package.
	 * @param dt the date/time the package was generated by SimBrief
	 */
	public void setCreatedOn(Instant dt) {
		_createdOn = dt;
	}
	
	/**
	 * Updates the SimBrief plan ID.
	 * @param id the SimBrief ID
	 */
	public void setSimBriefID(String id) {
		_sbID = id;
	}
	
	/**
	 * Updates the SimBrief release version.
	 * @param v the version
	 */
	public void setReleaseVersion(int v) {
		_releaseVersion = v;
	}
	
	/**
	 * Updates the SimBrief User ID.
	 * @param id the SimBrief user ID
	 */
	public void setSimBriefUserID(String id) {
		_sbUserID = id;
	}
	
	/**
	 * Updates the SimBrief request ID.
	 * @param id the SimBrief request ID
	 */
	public void setRequestID(String id) {
		_sbReqID = id;
	}
	
	/**
	 * Updates the AIRAC version used for this route.
	 * @param airac the AIRAC version
	 */
	public void setAIRAC(int airac) {
		_airac = airac;
	}
	
	/**
	 * Adds a destination alternate Airport.
	 * @param a the Airport
	 */
	public void addAirportL(Airport a) {
		_alternates.add(a);
	}
	
	/**
	 * Adds an ETOPS alternate Airport.
	 * @param a the Airport
	 */
	public void addETOPSAlternate(Airport a) {
		_etopsAlts.add(a);
	}
	
	/**
	 * Sets the ETOPS equal time midpoint.
	 * @param loc
	 */
	public void setETOPSMidpoint(GeoLocation loc) {
		_etopsMidpoint = loc;
	}
	
	/**
	 * Updates the cruise altitude.
	 * @param alt the altitude in feet
	 */
	public void setCruiseAltitude(String alt) {
		_cruiseAlt = alt;
	}
	
	/**
	 * Updates the flight route.
	 * @param rt the route
	 */
	public void setRoute(String rt) {
		_route = rt;
	}
	
	/**
	 * Updates the route's ETOPS rating.
	 * @param e an ETOPS
	 */
	public void setETOPS(ETOPS e) {
		_etops = e;
	}
	
	/**
	 * Updates the aircraft tail code.
	 * @param tc the tail code
	 */
	public void setTailCode(String tc) {
		_tailCode = tc;
	}
	
	/**
	 * Updates the SimBrief custom airframe ID.
	 * @param id the ID
	 */
	public void setAirframeID(String id) {
		_airframeID = id;
	}
	
	/**
	 * Updates the departure Runway.
	 * @param rwy the departure runway name
	 */
	public void setRunwayD(String rwy) {
		_rwyD = rwy;
	}
	
	/**
	 * Updates the arrival Runway.
	 * @param rwy the arrival runway name
	 */
	public void setRunwayA(String rwy) {
		_rwyA = rwy;
	}
	
	/**
	 * Updates the URL used to download the briefing package.
	 * @param url the SimBrief URL
	 */
	public void setURL(String url) {
		_url = url;
	}
	
	/**
	 * Updates the base Flight Plan URL.
	 * @param url the URL
	 */
	public void setBasePlanURL(String url) {
		_baseFPURL = url;
	}
	
	/**
	 * Updates the raw XML generated by SimBrief.
	 * @param doc the XML Document
	 */
	public void setXML(Document doc) {
		_xml = doc;
	}
	
	/**
	 * Updates the base fuel load.
	 * @param fuel the fuel in pounds
	 */
	public void setBaseFuel(int fuel) {
		_baseFuel = fuel;
	}
	
	/**
	 * Updates the taxi fuel load.
	 * @param fuel the fuel in pounds
	 */
	public void setTaxiFuel(int fuel) {
		_taxiFuel = fuel;
	}

	/**
	 * Updates the enroute fuel load.
	 * @param fuel the fuel in pounds
	 */
	public void setEnrouteFuel(int fuel) {
		_enrouteFuel = fuel;
	}
	
	/**
	 * Updates the alternate fuel load.
	 * @param fuel the fuel in pounds
	 */
	public void setAlternateFuel(int fuel) {
		_altFuel = fuel;
	}

	/**
	 * Updates the number of passengers booked on this flight.
	 * @param pax the number of passengers
	 */
	public void setPax(int pax) {
		_pax = pax;
	}
	
	/**
	 * Updates the total baggage weight for this flight.
	 * @param wt the weight in pounds
	 */
	public void setBaggageWeight(int wt) {
		_baggageWeight = wt;
	}
	
	/**
	 * Updates the per-passenger baggage weight for this flight.
	 * @param wt the weight in pounds
	 */
	public void setBagWeight(int wt) {
		_bagWeight = wt;
	}
	
	/**
	 * Updates the additional cargo weight for this flight.
	 * @param wt the weight in pounds
	 */
	public void setCargoWeight(int wt) {
		_cargoWeight = wt;
	}
}