// Copyright 2009, 2011, 2012, 2016, 2020, 2023 Global Virtual Airlines Group. All Rights Reserved.
package org.deltava.beans.wx;

import java.time.*;
import java.time.temporal.*;
import java.time.format.DateTimeFormatterBuilder;

import java.util.*;

import org.apache.logging.log4j.*;

import org.deltava.beans.navdata.AirportLocation;
import org.deltava.beans.wx.CloudLayer.Amount;
import org.deltava.beans.wx.Condition.Intensity;
import org.deltava.beans.wx.Condition.Qualifier;

import org.deltava.util.*;

/**
 * A parser for METAR data. 
 * @author Luke
 * @version 11.1
 * @since 2.6
 */

public class MetarParser {
	
	private static final Logger log = LogManager.getLogger(MetarParser.class);
	
	// Private
	private MetarParser() {
		super();
	}
	
	/**
	 * Parses a METAR.
	 * @param data the METAR data
	 * @return the parsed METAR data, or null if unparseable
	 */
	public static METAR parse(String data) {
		METAR result = new METAR();
		
		boolean isRemark = false;
		StringBuilder rmk = new StringBuilder();
		
		StringTokenizer st = new StringTokenizer(data, " ");
		String token = null;
		//String lastToken = null; 
		while (st.hasMoreTokens()) {
			token = st.nextToken();
			
			// Get station name
			if (result.getCode() == null) {
				//result.setAirport(SystemData.getAirport(token));
				//if (result.getCode() == null) {
					AirportLocation ap = new AirportLocation(0, 0);
					ap.setCode(token);
					result.setAirport(ap);
				//}
			// TODO: currently ignoring everything in the remark and there is some good data in there...
			} else if (token.equals("RMK")) {
				isRemark = true;
				//result.setRemarks(token);
				break;
			// Check if corrected
			} else if (token.equals("NOSIG")) {
				isRemark = false;
			} else if (token.equals("AUTO")) {
				isRemark = false;
				result.setIsAutoGenerated(true);
			} else if (token.equals("COR")) {
				isRemark = false;
				result.setIsCorrected(true);
			} else if (token.equals("CAVOK")) {
				isRemark = false;
				result.setVisibility(Distance.SM.getFeet(50), false);

			// Parse date
			} else if (isDate(token)) {
				int endOfs = token.endsWith("Z") ? 1 : 0;
				ZonedDateTime zdt = ZonedDateTime.now(ZoneOffset.UTC).truncatedTo(ChronoUnit.DAYS).withDayOfMonth(1);
				DateTimeFormatterBuilder dfb = new DateTimeFormatterBuilder().appendPattern("ddHHmm");
				dfb.parseDefaulting(ChronoField.MONTH_OF_YEAR, zdt.get(ChronoField.MONTH_OF_YEAR));
				dfb.parseDefaulting(ChronoField.YEAR_OF_ERA, zdt.get(ChronoField.YEAR));
				LocalDateTime ldt = LocalDateTime.parse(token.substring(0, token.length() - endOfs), dfb.toFormatter());
				result.setDate(ldt.toInstant(ZoneOffset.UTC));
			} else if (isWind(token)) {
				isRemark = false;
				if (token.charAt(0) == 'V')
					result.setWindDirection(-1);
				else
					result.setWindDirection(StringUtils.parse(token.substring(0, 3), 0));
				
				int pos = 3;
				while (Character.isDigit(token.charAt(pos))) pos++;
				result.setWindSpeed(StringUtils.parse(token.substring(3, pos), 0));
				if (token.charAt(pos) == 'G') {
					pos++; int pos2 = pos;
					while (Character.isDigit(token.charAt(pos2))) pos2++;
					result.setWindGust(StringUtils.parse(token.substring(pos, pos2), 0));
				}
				
				// Convert from MPS to KTS
				if (token.endsWith("MPS")) {
					result.setWindSpeed(Math.round(result.getWindSpeed() * 1.9438f));
					result.setWindGust(Math.round(result.getWindGust() * 1.9438f));
				}
			} else if (isTemperature(token)) {
				isRemark = false;
				boolean minus = (token.charAt(0) == 'M');
				int pos = minus ? 1 : 0; int pos2;
				for (pos2 = pos; token.charAt(pos2) != '/'; pos2++) { /* empty */ }
				result.setTemperature(StringUtils.parse(token.substring(pos, pos2), 0) * (minus ? -1 : 1));
				
				minus = (token.charAt(++pos2) == 'M');
				pos = pos2 + (minus ? 2 : 1);
				result.setDewPoint(StringUtils.parse(token.substring(pos), 0) * (minus ? -1 : 1));
				
				// Calculate RH
				if ((result.getTemperature() != 0) && (result.getDewPoint() != 0)) {
					double temp1 = 6.11 * Math.pow(10.0, (7.5 * result.getTemperature()) / (237.7 + result.getTemperature()));
					double temp2 = 6.11 * Math.pow(10.0, (7.5 * result.getDewPoint()) / (237.7 + result.getDewPoint()));
					result.setHumidity((int)Math.round(temp2 * 100 / temp1));
				}
			} else if (isCondition(token)) {
				isRemark = false;
				Intensity i = Intensity.NONE;
				if (token.charAt(0) == '-')
					i = Intensity.LIGHT;
				else if (token.charAt(0) == '+')
					i = Intensity.HEAVY;
				
				Condition c = null; Qualifier q = Qualifier.NONE;
				int pos = (i == Intensity.NONE) ? 0 : 1;
				while ((pos + 2) <= token.length()) {
					String cnd = token.substring(pos, pos +2);
					try {
						Condition.Type t = Condition.Type.valueOf(cnd);
						c = new Condition(t);
						c.setQualifier(q);
					} catch (IllegalArgumentException iae) {
						try {
							q = Qualifier.valueOf(cnd);
							if (c != null)
								c.setQualifier(q);
						} catch (Exception e) {
							q = Qualifier.NONE;
						}
					}
					
					pos += 2;
				}
				
				if (c != null)
					result.add(c);
			} else if (isCloud(token)) {
				isRemark = false;
				Amount a = Amount.VV;
				int pos = 2;
				if (!token.startsWith(Amount.VV.toString())) {
					a = Amount.valueOf(token.substring(0, 3));
				    pos++;
				}
				
				int h = 0;
				if ((pos+3) <= token.length()) {
					h = StringUtils.parse(token.substring(pos, (pos+3)), 0);
					pos  += 3;
				}
				
				CloudLayer cl = new CloudLayer(h * 100);
				cl.setThickness(a);
				if (pos < token.length()) 
					cl.setType(CloudLayer.Type.valueOf(token.substring(pos)));
				
				if (a.ordinal() > Amount.CLR.ordinal())
					result.add(cl);
			} else if (isRVR(token)) {
				int pos = token.indexOf('/');
				RunwayVisualRange rvr = new RunwayVisualRange(token.substring(1, pos));
				String tmpToken = token.substring(pos + 1, token.length() - 2);
				pos = tmpToken.indexOf('V');
				if (pos != -1) {
					rvr.setMinVisibility(StringUtils.parse(tmpToken.substring(0, pos), 0));
					rvr.setMaxVisibility(StringUtils.parse(tmpToken.substring(pos + 1), 0));
				} else
					rvr.setMinVisibility(StringUtils.parse(tmpToken, 0));
				
				result.add(rvr);
			} else if (isAltimeter(token)) {
				isRemark = false;
				if (token.charAt(0) == 'A')
					result.setPressure(StringUtils.parse(token.substring(1, 3), 0.0d) +
						(StringUtils.parse(token.substring(3, 5), 0.0d) / 100));
				else {
					double hp = StringUtils.parse(token.substring(1), 0.0d);
					result.setPressure(hp * 0.02952875d);
				}
			} else if (isVisibility(token, result.getCode())) {
				isRemark = false;
				boolean lessThan = (token.charAt(0) == 'M');
				Distance dst = Distance.METERS; int ePos = 0;
				if (token.endsWith("SM")) {
					ePos = 2;
					dst = Distance.SM;
				} else if (token.endsWith("M"))
					ePos = 1;
				
				String tmpToken = token.substring(lessThan ? 1 : 0, token.length() - ePos);
				int pos = tmpToken.indexOf('/');
				if (pos > -1) {
					double viz = StringUtils.parse(tmpToken.substring(0, pos), -1.0d) /
						StringUtils.parse(tmpToken.substring(pos + 1), 1.0d);
					if (viz < 0) {
						log.warn("Unparseable visibility - {}", token);
						result.setVisibility(Distance.SM.getFeet(15), false);
					} else
						result.setVisibility(dst.getFeet(viz), lessThan);
				} else {
					int viz = StringUtils.parse(tmpToken, -1);
					if (viz == -1) {
						log.warn("Unparseable visibility - {}", token);
						result.setVisibility(Distance.SM.getFeet(15), false);
					} else
						result.setVisibility(dst.getFeet(viz), lessThan);
				}
			} else if (isRemark) {
				rmk.append(token);
				rmk.append(' ');
			} else if (log.isDebugEnabled())
				log.debug("Unknown METAR token - {}", token);
			
			//lastToken = token;
		}
		
		if (rmk.length() > 0)
			result.setRemarks(rmk.toString());
		
		return result;
	}
	
	private static boolean isDate(String date) {
		return (!StringUtils.isEmpty(date) && date.matches("\\d{6}Z"));
	}
	
	private static boolean isWind(String tkn) {
		char[] buf = tkn.toCharArray();
		if (buf.length < 7)
			return false;
		if (!tkn.endsWith("KT") && !tkn.endsWith("MPS"))
			return false;
		
		// Wind: 3 digit true-north direction,
		// nearest 10 degrees (or VaRiaBle);
		if (buf[0] == 'V') {
			if ((buf[1] != 'R') || (buf[2] != 'B')) {
				return false;
			}
		} else if ((!Character.isDigit(buf[0])) || (!Character.isDigit(buf[1]))
				|| (!Character.isDigit(buf[2])))
			return false;
		
		// next 2-3 digits for speed
		if ((!Character.isDigit(buf[3])) || (!Character.isDigit(buf[4])))
			return false;
		
		int i = 5;
		if (Character.isDigit(buf[i]))
			i++;
		
		// Check for gusts
		if (buf[i] == 'G') {
			int j;
			for (j = 1; Character.isDigit(buf[i + j]) && (i + j) < buf.length; j++) { /* empty */ }
			if (j == 3 || j == 4)
				i += j;
			else
				return false;
		}

		return true;
	}

	private static boolean isTemperature(String tkn) {
		char[] buf = tkn.toCharArray();
		if (buf.length < 5)
			return false;
		
		int pos = 0;
		if (buf[pos] == 'M') 
			pos++;
		
		if ((!Character.isDigit(buf[pos])) || (!Character.isDigit(buf[++pos]))) 
			return false;

		if (buf[++pos] != '/') 
			return false;

		if (buf[++pos] == 'M') 
			pos++;

		if ((!Character.isDigit(buf[pos])) || (! Character.isDigit(buf[++pos]))) 
			return false;
		
		return true;
	}

	private static boolean isCondition(String tkn) {
		char[] buf = tkn.toCharArray();
		if (buf.length < 2 || buf.length > 8)
			return false;
		
		int pos = 0;
		if ((buf[0] == '-' || buf[0] == '+') && !tkn.startsWith("+FC"))
			pos++;
		
		boolean found = false;
		while ((pos + 2) <= buf.length) {
			found = false;
			if (Condition.isValid(tkn.substring(pos, pos +2))) {
				found = true;
				pos += 2;
			}
			
			// look for the odd tornado/waterspout token along the way
			if (((pos + 3) <= buf.length) && tkn.substring(pos, pos + 3).equals("+FC")) {
				found = true;
				pos += 3;
			}

			if (!found)
				break;
		}
		
		// if we didn't match all the tokens its not a condition...
		return (pos >= tkn.length());
	}
	
	private static boolean isCloud(String tkn) {
		if (tkn.equals("SKC") || tkn.equals("CLR"))
			return true;
		
		int pos = 0;
		if (tkn.startsWith(Amount.VV.toString()))
			pos = 2;
		else if (tkn.startsWith(Amount.FEW.toString()) || tkn.startsWith(Amount.SCT.toString()) ||
			    tkn.startsWith(Amount.BKN.toString()) || tkn.startsWith(Amount.OVC.toString()))
			pos = 3;
		else
			return false;
		
		if ((pos+3) <= tkn.length()) {
			int endIndex = pos+3;
			for (int j = pos; j < endIndex; j++) {
				if (!Character.isDigit(tkn.charAt(j)))
					return false;
			}
			pos += 3;
		} else
			return false;
		
		if (pos == tkn.length()) 
			return true;

		String cType = tkn.substring(pos);
		return (cType.equals(CloudLayer.Type.TCU.toString()) || cType.equals(CloudLayer.Type.CB.toString()));
	}

	private static boolean isAltimeter(String tkn) {
		if (tkn.length() != 5)
			return false;
		else if (tkn.charAt(0) != 'A' && tkn.charAt(0) != 'Q')
			return false;

		for (int i = 1; i < 5; i++)
			if (!Character.isDigit(tkn.charAt(i))) return false;

		return true;
	}

	private static boolean isVisibility(String tkn, String apCode) {
		
		// must have at least one digit and end in "SM", or be only four digits
		if ((tkn.length() < 3) && !tkn.endsWith("SM"))
			return false;
		else if (!tkn.endsWith("SM")) {
			boolean isNA = (apCode.startsWith("C") || apCode.startsWith("K") || apCode.startsWith("PA")
					|| apCode.startsWith("PG") || apCode.startsWith("MM") || apCode.startsWith("TJ"));
			if ((tkn.length() > 2) && (tkn.length() < 6) && !isNA) {
				int endPos = tkn.endsWith("M") ? 1 : 0;
				for (int j = 0; j < (tkn.length() - endPos); j++) {
					if (!Character.isDigit(tkn.charAt(j)))
						return false;
				}
				
				return true;
			}
			
			return false;
		}
		
		char lastChar = '0';
		boolean foundDigit = false, foundSpace = false, foundFraction = false;
		
		// examine everything in front of the token ending "SM"
		int pos = (tkn.charAt(0) == 'M') ? 1 : 0;
		for (; pos < (tkn.length() - 2); pos++) {
			if (Character.isDigit(tkn.charAt(pos))) {
				foundDigit = true;
			} else if (tkn.charAt(pos) == ' ') {
				// only permit a single space and can't be followed by '/'
				if (foundSpace || lastChar == '/')
					return false;
				foundDigit = false;
				foundSpace = true;
			} else if (tkn.charAt(pos) == '/') {
				// only permit a single fraction and can't be followed by ' '
				if (foundFraction || lastChar == ' ')
					return false;
				foundDigit = false;
				foundFraction = true;
			} else
				return false;
			
			lastChar = tkn.charAt(pos);
		}
		
		// can't be left needing a digit...
		return foundDigit;
	}
	
	private static boolean isRVR(String tkn) {
		return (tkn.startsWith("R") && tkn.contains("/") && (tkn.length() > 4) && !tkn.endsWith("FT") && (tkn.indexOf('/') < (tkn.length()-2)));
	}
}